<?php
/* --------------------------------------------------------------
   OrderDetails.php 2020-01-27
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2020 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Admin\Withdrawal\Models;

use DateTime;
use Exception;
use Gambio\Admin\Withdrawal\Interfaces\WithdrawalOrderId;
use InvalidArgumentException;

/**
 * Class OrderDetails
 *
 * @package Gambio\Admin\Withdrawal\Models
 */
class OrderDetails
{
    /**
     * @var WithdrawalOrderId
     */
    private $id;
    
    /**
     * @var DateTime|null
     */
    private $creationDate;
    
    /**
     * @var DateTime|null
     */
    private $deliveryDate;
    
    
    /**
     * OrderDetails constructor.
     *
     * @param WithdrawalOrderId $id
     * @param DateTime|null     $creationDate
     * @param DateTime|null     $deliveryDate
     */
    private function __construct(WithdrawalOrderId $id, ?DateTime $creationDate, ?DateTime $deliveryDate)
    {
        $this->id           = $id;
        $this->creationDate = $creationDate;
        $this->deliveryDate = $deliveryDate;
    }
    
    
    /**
     * Create a new OrderDetails instance.
     *
     * @param WithdrawalOrderId $id
     * @param DateTime|null     $creationDate
     * @param DateTime|null     $deliveryDate
     *
     * @return OrderDetails
     */
    public static function create(
        WithdrawalOrderId $id,
        ?DateTime $creationDate,
        ?DateTime $deliveryDate
    ): OrderDetails {
        return new self($id, $creationDate, $deliveryDate);
    }
    
    
    /**
     * Returns the ID.
     *
     * @return int|null
     */
    public function id(): ?int
    {
        return $this->id->value();
    }
    
    
    /**
     * Returns the create date.
     *
     * @return string|null
     */
    public function creationDate(): ?string
    {
        return ($this->creationDate !== null) ? $this->creationDate->format('Y-m-d H:i:s') : null;
    }
    
    
    /**
     * Returns the delivery date.
     *
     * @return string|null
     */
    public function deliveryDate(): ?string
    {
        return ($this->deliveryDate !== null) ? $this->deliveryDate->format('Y-m-d H:i:s') : null;
    }
    
    
    /**
     * Changes the related order of the withdrawal.
     *
     * @param WithdrawalOrderId $orderId
     */
    public function changeId(WithdrawalOrderId $orderId): void
    {
        $this->id = $orderId;
    }
    
    
    /**
     * Array serialization.
     *
     * @return array
     */
    public function toArray(): array
    {
        return [
            'id'           => $this->id(),
            'creationDate' => $this->creationDate(),
            'deliveryDate' => $this->deliveryDate(),
        ];
    }
    
    
    /**
     * Create a new OderDetails instance from the provided array.
     *
     * @param array $data
     *
     * @return OrderDetails
     */
    public static function createFromArray(array $data): OrderDetails
    {
        try {
            $creationDate = ($data['creationDate'] !== null) ? new DateTime($data['creationDate']) : null;
        } catch (Exception $e) {
            throw new InvalidArgumentException('Invalid datetime format for order creation date provided.');
        }
        try {
            $deliveryDate = ($data['deliveryDate'] !== null) ? new DateTime($data['deliveryDate']) : null;
        } catch (Exception $e) {
            throw new InvalidArgumentException('Invalid datetime format for order delivery date provided.');
        }
        
        return OrderDetails::create(OrderId::create($data['id']), $creationDate, $deliveryDate);
    }
}